# Monorepo Uitleg - Wat is het en hoe werkt het?

## 🤔 Wat is een Monorepo?

Een **monorepo** (mono repository) is één Git repository die meerdere projecten of codebases bevat. In jouw geval:

- **Normaal**: Elke VPS heeft zijn eigen aparte Git repository
- **Met Monorepo**: Beide VPS'en delen één Git repository, maar hebben elk hun eigen codebase

## 📁 Jouw Situatie

Je hebt twee VPS servers:
- **VPS1 (Adalyapoint)**: Vape webshop - hoofdwebsite
- **VPS2 (Obevaria)**: Furniture shop - betalingsverwerking

Beide systemen zijn bijna identiek qua code, maar hebben:
- Verschillende configuraties (database, URLs)
- Verschillende functionaliteit (vape vs furniture)
- Verschillende databases

## 🎯 Waarom een Monorepo?

### Voordelen:

1. **Eenvoudige Synchronisatie**
   - Fix op VPS2? Direct beschikbaar op VPS1 via `git pull`
   - Geen handmatig kopiëren van bestanden meer

2. **Één Bron van Waarheid**
   - Alle code staat op één plek
   - Duidelijk welke versie waar draait

3. **Gedeelde Wijzigingen**
   - Bugfix in één keer voor beide systemen
   - Nieuwe features makkelijk te delen

4. **Versiecontrole**
   - Volledige geschiedenis van alle wijzigingen
   - Makkelijk terugdraaien van fouten

### Nadelen:

1. **Configuratie moet apart blijven**
   - Elke VPS heeft zijn eigen database configuratie
   - Daarom staan `database.php` en `app.php` in `.gitignore`

2. **Voorzichtig met pushen**
   - Test altijd eerst op één VPS
   - Push niet zomaar alles naar beide

## 🏗️ Hoe ziet de Structuur eruit?

```
/var/www/html/                    (Git repository root)
│
├── .git/                         (Git configuratie - NIET aanraken)
├── .gitignore                    (Welke bestanden NIET committen)
├── README.md                     (Documentatie)
│
├── Oraviare/                     (VPS2 - Obevaria codebase)
│   └── Admin/
│       ├── app/
│       ├── config/
│       │   ├── database.php      ❌ NIET in Git (VPS-specifiek)
│       │   └── app.php           ❌ NIET in Git (VPS-specifiek)
│       ├── public/
│       └── ...
│
└── AdalyaPoint/                  (VPS1 - Adalyapoint codebase)
    └── Admin/
        ├── app/
        ├── config/
        │   ├── database.php      ❌ NIET in Git (VPS-specifiek)
        │   └── app.php           ❌ NIET in Git (VPS-specifiek)
        ├── public/
        └── ...
```

## 🔄 Hoe werkt de Workflow?

### Scenario 1: Bugfix op VPS2, ook nodig op VPS1

**Op VPS2:**
```bash
cd /var/www/html
# Maak de fix
nano Oraviare/Admin/app/Controllers/DashboardController.php

# Commit en push
git add Oraviare/Admin/app/Controllers/DashboardController.php
git commit -m "Fix: Dashboard connection check"
git push
```

**Op VPS1:**
```bash
cd /var/www/html
# Haal de wijziging op
git pull

# De fix is nu ook op VPS1! ✅
```

### Scenario 2: Nieuwe feature toevoegen

**Op VPS2:**
```bash
cd /var/www/html
# Maak nieuwe feature
# ... code wijzigingen ...

git add .
git commit -m "Feature: Nieuwe betalingsmethode"
git push
```

**Op VPS1:**
```bash
cd /var/www/html
git pull
# Feature is nu beschikbaar op beide VPS'en! ✅
```

### Scenario 3: Alleen VPS2-specifieke wijziging

**Op VPS2:**
```bash
cd /var/www/html
# Wijzig alleen Oraviare code
# ... code wijzigingen ...

git add Oraviare/
git commit -m "VPS2 only: Furniture category update"
git push
```

**Op VPS1:**
```bash
cd /var/www/html
git pull
# Alleen Oraviare/ is bijgewerkt, AdalyaPoint/ blijft hetzelfde ✅
```

## ⚙️ Belangrijke Bestanden

### `.gitignore` - Wat wordt NIET gecommit?

Dit bestand vertelt Git welke bestanden **niet** meegenomen moeten worden:

```
**/config/database.php    ❌ Elke VPS heeft zijn eigen database
**/config/app.php         ❌ Elke VPS heeft zijn eigen app config
**/.env                   ❌ Environment variabelen (gevoelig)
**/public/uploads/        ❌ Geüploade bestanden (te groot)
*.log                    ❌ Log bestanden
```

**Waarom?**
- Elke VPS heeft een andere database naam, wachtwoord, URL
- Deze moeten lokaal blijven en niet gedeeld worden

### Configuratie Bestanden

**Op VPS2:**
- `Oraviare/Admin/config/database.php` → Database: `oraviare`
- `Oraviare/Admin/config/app.php` → URL: `oraviare.com`

**Op VPS1:**
- `AdalyaPoint/Admin/config/database.php` → Database: `adyalapoint`
- `AdalyaPoint/Admin/config/app.php` → URL: `adyalapoint.com`

Deze bestanden staan in `.gitignore`, dus worden **niet** gesynchroniseerd.

## 🚀 Praktische Voorbeelden

### Voorbeeld 1: Controller Fix

**Probleem**: Er zit een bug in `DashboardController.php` die op beide VPS'en voorkomt.

**Oplossing**:
1. Fix op VPS2 in `Oraviare/Admin/app/Controllers/DashboardController.php`
2. Commit en push
3. Op VPS1: `git pull`
4. Fix is nu ook op VPS1 in `AdalyaPoint/Admin/app/Controllers/DashboardController.php`

### Voorbeeld 2: Nieuwe Model Toevoegen

**Taak**: Voeg een nieuwe `ProductModel.php` toe die beide VPS'en nodig hebben.

**Oplossing**:
1. Maak bestand op VPS2: `Oraviare/Admin/app/Models/ProductModel.php`
2. Kopieer naar VPS1 structuur: `AdalyaPoint/Admin/app/Models/ProductModel.php`
3. Commit beide:
   ```bash
   git add Oraviare/Admin/app/Models/ProductModel.php
   git add AdalyaPoint/Admin/app/Models/ProductModel.php
   git commit -m "Add: ProductModel voor beide systemen"
   git push
   ```

### Voorbeeld 3: Database Configuratie Aanpassen

**Taak**: Database wachtwoord wijzigen op VPS2.

**Oplossing**:
1. Pas aan: `Oraviare/Admin/config/database.php`
2. **NIET committen!** (staat in `.gitignore`)
3. Alleen lokaal op VPS2, VPS1 blijft ongewijzigd ✅

## 🔐 Veiligheid

### Wat is veilig om te committen?

✅ **Code bestanden**: PHP, JavaScript, CSS, HTML
✅ **Database schema's**: SQL bestanden (zonder data)
✅ **Documentatie**: README, instructies
✅ **Configuratie templates**: `.example` bestanden

### Wat is NIET veilig om te committen?

❌ **Wachtwoorden**: Database wachtwoorden, API keys
❌ **Persoonlijke data**: Gebruikersdata, orders
❌ **Uploads**: Geüploade bestanden
❌ **Logs**: Error logs met gevoelige informatie

## 📊 Git Commands Overzicht

### Basis Commands

```bash
# Status bekijken
git status

# Wijzigingen bekijken
git diff

# Bestanden toevoegen
git add <bestand>
git add .                    # Alle wijzigingen

# Commit maken
git commit -m "Beschrijving"

# Push naar remote
git push

# Pull van remote
git pull

# Geschiedenis bekijken
git log --oneline
```

### Handige Commands

```bash
# Alleen specifieke directory
git add Oraviare/

# Laatste commit ongedaan maken (lokaal)
git reset --soft HEAD~1

# Wijzigingen weggooien (voorzichtig!)
git checkout -- <bestand>

# Branch maken (voor experimenten)
git checkout -b experiment
```

## 🎓 Samenvatting

**Monorepo = Eén Git repo voor beide VPS'en**

**Voordelen:**
- ✅ Eenvoudige synchronisatie
- ✅ Gedeelde bugfixes
- ✅ Één bron van waarheid

**Belangrijk:**
- ⚠️ Configuratie bestanden blijven lokaal (`.gitignore`)
- ⚠️ Test altijd eerst op één VPS
- ⚠️ Commit alleen code, geen gevoelige data

**Workflow:**
1. Maak wijziging op VPS2
2. `git add .` en `git commit -m "..."` en `git push`
3. Op VPS1: `git pull`
4. Klaar! ✅

## ❓ Veelgestelde Vragen

**V: Wat als ik per ongeluk een wachtwoord commit?**
A: Direct verwijderen uit Git geschiedenis, wachtwoord wijzigen, en `.gitignore` controleren.

**V: Kan ik alleen VPS2 code updaten zonder VPS1?**
A: Ja, commit alleen `Oraviare/` directory. VPS1 krijgt alleen updates als je `git pull` doet.

**V: Wat als beide VPS'en tegelijk wijzigen?**
A: Git merge conflicten kunnen ontstaan. Los op door eerst te pullen, dan te committen.

**V: Moet ik altijd beide directories updaten?**
A: Nee, alleen als de wijziging voor beide systemen geldt. Anders alleen de relevante directory.

**V: Hoe weet ik welke versie waar draait?**
A: `git log` toont de commit geschiedenis. Elke VPS heeft dezelfde commits na `git pull`.
