<?php
namespace App\Controllers;

use Core\Controller;
use App\Models\FurnitureOrder;

/**
 * CheckoutController
 * Publieke betalingspagina voor orders
 */
class CheckoutController extends Controller
{
    private FurnitureOrder $orderModel;

    public function __construct()
    {
        parent::__construct();
        $this->orderModel = new FurnitureOrder();
    }

    /**
     * GET /checkout/{id}
     * Toon betalingspagina voor furniture order
     */
    public function show(string $id): void
    {
        // Haal order op
        $order = $this->orderModel->findByFurnitureOrderId($id);
        
        if (!$order) {
            http_response_code(404);
            die('Order niet gevonden');
        }

        // Check of payment URL nog geldig is
        if ($order['payment_url_expires_at'] && strtotime($order['payment_url_expires_at']) < time()) {
            $this->view('checkout/expired', [
                'order' => $order,
                'title' => 'Betalingslink verlopen'
            ]);
            return;
        }

        // Toon betalingspagina
        $title = 'Afrekenen';
        $contentView = 'checkout/show';
        extract(['order' => $order]);
        require __DIR__ . '/../Views/layouts/public.php';
    }

    /**
     * POST /checkout/{id}/pay
     * Verwerk test betaling
     */
    public function pay(string $id): void
    {
        // Haal order op
        $order = $this->orderModel->findByFurnitureOrderId($id);
        
        if (!$order) {
            http_response_code(404);
            die('Order niet gevonden');
        }

        // Update payment status naar paid
        $this->orderModel->updatePaymentStatus($id, 'paid', [
            'transaction_id' => 'TEST-' . time(),
            'paid_at' => date('Y-m-d H:i:s')
        ]);

        // Stuur webhook naar VPS1
        try {
            $webhookService = new \App\Services\WebhookService();
            $webhookService->sendPaymentStatusUpdate($id, 'paid', [
                'transaction_id' => 'TEST-' . time(),
                'paid_at' => date('Y-m-d H:i:s')
            ]);
        } catch (\Exception $e) {
            error_log("Webhook error: " . $e->getMessage());
        }

        // Redirect naar success pagina
        $this->redirect('/checkout/' . $id . '/success');
    }

    /**
     * GET /checkout/{id}/success
     * Toon success pagina na betaling
     */
    public function success(string $id): void
    {
        $order = $this->orderModel->findByFurnitureOrderId($id);
        
        if (!$order) {
            http_response_code(404);
            die('Order niet gevonden');
        }

        $title = 'Betaling voltooid';
        $contentView = 'checkout/success';
        extract(['order' => $order]);
        require __DIR__ . '/../Views/layouts/public.php';
    }
}
