<?php
namespace App\Controllers;

use Core\Controller;
use App\Models\FurnitureOrder;
use App\Models\FurnitureProduct;
use App\Models\ApiCredential;
use App\Services\WebhookService;

/**
 * OrderController
 * Beheert orders op VPS2 voor monitoring
 */
class OrderController extends Controller
{
    private FurnitureOrder $orderModel;
    private FurnitureProduct $productModel;
    private ApiCredential $apiCredentialModel;
    private WebhookService $webhookService;

    public function __construct()
    {
        parent::__construct();
        $this->orderModel = new FurnitureOrder();
        $this->productModel = new FurnitureProduct();
        $this->apiCredentialModel = new ApiCredential();
        $this->webhookService = new WebhookService();
    }

    /**
     * Overzicht van alle orders
     */
    public function index(): void
    {
        $orders = $this->orderModel->getAllWithDetails();
        
        // Statistieken
        $stats = [
            'total' => count($orders),
            'pending' => count(array_filter($orders, fn($o) => $o['payment_status'] === 'pending')),
            'paid' => count(array_filter($orders, fn($o) => $o['payment_status'] === 'paid')),
            'failed' => count(array_filter($orders, fn($o) => $o['payment_status'] === 'failed')),
            'pending_webhooks' => count(array_filter($orders, fn($o) => !$o['webhook_sent'] && in_array($o['payment_status'], ['paid', 'failed', 'cancelled'])))
        ];

        $this->json([
            'success' => true,
            'orders' => $orders,
            'stats' => $stats
        ]);
    }

    /**
     * Order details
     */
    public function show(string $id): void
    {
        $order = $this->orderModel->findByFurnitureOrderId($id);

        if (!$order) {
            $this->json([
                'success' => false,
                'error' => 'Not found',
                'message' => 'Order not found'
            ], 404);
            return;
        }

        $this->json([
            'success' => true,
            'order' => $order
        ]);
    }

    /**
     * Handmatig webhook versturen
     */
    public function sendWebhook(string $id): void
    {
        $order = $this->orderModel->findByFurnitureOrderId($id);

        if (!$order) {
            $this->json([
                'success' => false,
                'error' => 'Not found',
                'message' => 'Order not found'
            ], 404);
            return;
        }

        $success = $this->webhookService->sendPaymentStatusUpdate(
            $id,
            $order['payment_status'],
            [
                'transaction_id' => $order['transaction_id'],
                'paid_at' => $order['paid_at'],
                'failure_reason' => $order['failure_reason']
            ]
        );

        if ($success) {
            $this->json([
                'success' => true,
                'message' => 'Webhook sent successfully'
            ]);
        } else {
            $this->json([
                'success' => false,
                'error' => 'Failed to send webhook',
                'message' => 'Webhook could not be sent'
            ], 500);
        }
    }

    /**
     * Process alle pending webhooks
     */
    public function processPendingWebhooks(): void
    {
        $sent = $this->webhookService->processPendingWebhooks();

        $this->json([
            'success' => true,
            'message' => "Processed {$sent} pending webhooks"
        ]);
    }
}

