<?php
namespace App\Models;

use Core\Model;

/**
 * FurnitureCategory Model
 * Beheert furniture product categorieën
 */
class FurnitureCategory extends Model
{
    protected string $table = 'furniture_categories';

    /**
     * Alle actieve categorieën ophalen
     */
    public function getAllActive(): array
    {
        return $this->query(
            "SELECT * FROM {$this->table} WHERE status = 'active' ORDER BY sort_order ASC, name ASC"
        );
    }

    /**
     * Categorie ophalen op basis van slug
     */
    public function findBySlug(string $slug): ?array
    {
        return $this->where('slug', $slug);
    }

    /**
     * Categorieën met product count
     */
    public function getAllWithProductCount(): array
    {
        return $this->query(
            "SELECT c.*, COUNT(p.id) as product_count
             FROM {$this->table} c
             LEFT JOIN furniture_products p ON c.name = p.category AND p.is_active = 1
             GROUP BY c.id
             ORDER BY c.sort_order ASC, c.name ASC"
        );
    }

    /**
     * Check of slug al bestaat
     */
    public function slugExists(string $slug, ?int $excludeId = null): bool
    {
        $sql = "SELECT id FROM {$this->table} WHERE slug = ?";
        $params = [$slug];
        
        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }
        
        return $this->queryOne($sql, $params) !== null;
    }

    /**
     * Genereer slug van naam
     */
    public function generateSlug(string $name): string
    {
        $slug = strtolower($name);
        $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
        $slug = trim($slug, '-');
        
        // Check of slug al bestaat, voeg nummer toe indien nodig
        $originalSlug = $slug;
        $counter = 1;
        while ($this->slugExists($slug)) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }
        
        return $slug;
    }
}
