<?php
namespace App\Models;

use Core\Model;

/**
 * FurnitureOrder Model
 * Beheert furniture orders van VPS1
 */
class FurnitureOrder extends Model
{
    protected string $table = 'furniture_orders';

    /**
     * Order ophalen op furniture_order_id
     */
    public function findByFurnitureOrderId(string $furnitureOrderId): ?array
    {
        return $this->where('furniture_order_id', $furnitureOrderId);
    }

    /**
     * Order ophalen op external_order_id (van VPS1)
     */
    public function findByExternalOrderId(string $externalOrderId): ?array
    {
        return $this->where('external_order_id', $externalOrderId);
    }

    /**
     * Nieuwe furniture order aanmaken
     */
    public function createOrder(array $data): ?string
    {
        // Genereer furniture_order_id
        $furnitureOrderId = 'FURN-' . strtoupper(uniqid());
        
        $orderData = [
            'furniture_order_id' => $furnitureOrderId,
            'external_order_id' => $data['external_order_id'],
            'product_id' => $data['furniture_product_id'],
            'product_name' => $data['furniture_product_name'],
            'quantity' => $data['quantity'] ?? 1,
            'price' => $data['price'],
            'currency' => $data['currency'] ?? 'EUR',
            'payment_status' => 'pending',
            'customer_email' => $data['customer']['email'] ?? null,
            'customer_name' => $data['customer']['name'] ?? null,
            'return_url' => $data['return_url'] ?? null,
            'cancel_url' => $data['cancel_url'] ?? null,
            'webhook_url' => $data['webhook_url'] ?? null,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];

        $id = $this->create($orderData);
        
        if ($id) {
            // Haal de order op met furniture_order_id
            $order = $this->findByFurnitureOrderId($furnitureOrderId);
            return $order ? $order['furniture_order_id'] : null;
        }
        
        return null;
    }

    /**
     * Payment status updaten
     */
    public function updatePaymentStatus(string $furnitureOrderId, string $status, array $additionalData = []): bool
    {
        $updateData = [
            'payment_status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        if ($status === 'paid' && isset($additionalData['paid_at'])) {
            $updateData['paid_at'] = $additionalData['paid_at'];
        }

        if ($status === 'failed' && isset($additionalData['failed_at'])) {
            $updateData['failed_at'] = $additionalData['failed_at'];
            $updateData['failure_reason'] = $additionalData['failure_reason'] ?? null;
        }

        if ($status === 'cancelled') {
            // Cancelled status - kan cancelled_at timestamp toevoegen als nodig
            // Voor nu alleen status updaten
        }

        if (isset($additionalData['transaction_id'])) {
            $updateData['transaction_id'] = $additionalData['transaction_id'];
        }

        if (isset($additionalData['payment_url'])) {
            $updateData['payment_url'] = $additionalData['payment_url'];
        }

        if (isset($additionalData['payment_url_expires_at'])) {
            $updateData['payment_url_expires_at'] = $additionalData['payment_url_expires_at'];
        }

        return $this->db->execute(
            "UPDATE {$this->table} SET " . 
            implode(', ', array_map(fn($k) => "{$k} = ?", array_keys($updateData))) . 
            " WHERE furniture_order_id = ?",
            array_merge(array_values($updateData), [$furnitureOrderId])
        );
    }

    /**
     * Webhook status updaten
     */
    public function markWebhookSent(string $furnitureOrderId): bool
    {
        return $this->db->execute(
            "UPDATE {$this->table} SET webhook_sent = 1, webhook_sent_at = NOW() WHERE furniture_order_id = ?",
            [$furnitureOrderId]
        );
    }

    /**
     * Orders per status
     */
    public function getByPaymentStatus(string $status): array
    {
        return $this->whereAll('payment_status', $status);
    }

    /**
     * Orders die webhook nog niet verstuurd hebben
     */
    public function getPendingWebhooks(): array
    {
        return $this->query(
            "SELECT * FROM {$this->table} 
             WHERE webhook_sent = 0 
             AND payment_status IN ('paid', 'failed', 'cancelled')
             ORDER BY created_at ASC"
        );
    }

    /**
     * Alle orders met details (GEPAGINEERD)
     * FORCE INDEX om filesort te vermijden
     */
    public function getAllWithDetails(int $limit = 25, int $offset = 0): array
    {
        return $this->query(
            "SELECT * FROM {$this->table} 
             FORCE INDEX (idx_created_desc)
             ORDER BY created_at DESC
             LIMIT ? OFFSET ?",
            [$limit, $offset]
        );
    }

    /**
     * Tel totaal aantal orders
     */
    public function countAll(): int
    {
        $result = $this->queryOne("SELECT COUNT(*) as total FROM {$this->table}");
        return (int) ($result['total'] ?? 0);
    }

    /**
     * Tel orders per status
     */
    public function countByStatus(): array
    {
        $results = $this->query(
            "SELECT payment_status, COUNT(*) as count 
             FROM {$this->table} 
             GROUP BY payment_status"
        );
        
        $counts = [
            'pending' => 0,
            'paid' => 0,
            'failed' => 0,
            'cancelled' => 0
        ];
        
        foreach ($results as $row) {
            $counts[$row['payment_status']] = (int) $row['count'];
        }
        
        return $counts;
    }
}

