<?php
/**
 * VPS2 - Meubelwebshop Configuratie
 * Payment processing systeem voor VPS1 (Vape Webshop)
 */

// Detecteer automatisch de base URL op basis van de huidige request
if (!function_exists('detectBaseUrl')) {
    function detectBaseUrl(): string
    {
        // Als we via CLI draaien, gebruik default
        if (php_sapi_name() === 'cli') {
            return 'http://oraviare/Oraviare/Admin/public';
        }
        
        // Detecteer protocol (http of https)
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || 
                  (!empty($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443) 
                  ? 'https' : 'http';
        
        // Detecteer host (domeinnaam of localhost)
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // Detecteer base path (bijv. /Dashboard%202/public)
        $scriptName = $_SERVER['SCRIPT_NAME'] ?? '/index.php';
        $basePath = dirname($scriptName);
        
        // Normaliseer base path (verwijder trailing slash, behalve als het root is)
        $basePath = rtrim($basePath, '/');
        if ($basePath === '' || $basePath === '.') {
            $basePath = '';
        }
        
        return $scheme . '://' . $host . $basePath;
    }
}

/**
 * Detecteer VPS1 webhook URL automatisch
 * Probeert de webhook URL te bepalen op basis van:
 * 1. Laatste order die is aangemaakt (webhook_url uit database)
 * 2. Huidige request (als VPS1 en VPS2 opzelfde domein)
 * 3. Fallback naar localhost
 */
if (!function_exists('detectVps1WebhookUrl')) {
    function detectVps1WebhookUrl(): string
{
    // Als we via CLI draaien, gebruik default
    if (php_sapi_name() === 'cli') {
        return 'http://localhost/api/webhook/payment-status';
    }
    
    // Probeer VPS1 webhook URL op te halen uit database (uit laatste order)
    // Alleen als database config bestaat, PDO beschikbaar is en $_SERVER beschikbaar is
    if (isset($_SERVER) && class_exists('PDO') && file_exists(__DIR__ . '/database.php')) {
        try {
            // Laad database config
            $dbConfig = @require __DIR__ . '/database.php';
            
            if (is_array($dbConfig) && isset($dbConfig['host']) && isset($dbConfig['database'])) {
                // Maak database connectie met error suppression voor eerste keer
                $dsn = "mysql:host={$dbConfig['host']};dbname={$dbConfig['database']};charset=" . ($dbConfig['charset'] ?? 'utf8mb4');
                $options = $dbConfig['options'] ?? [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                ];
                
                $db = @new PDO($dsn, $dbConfig['username'] ?? 'root', $dbConfig['password'] ?? '', $options);
                
                if ($db) {
                    // Haal VPS1 webhook URL op uit de laatste order (als die bestaat)
                    $stmt = @$db->query("SELECT webhook_url FROM furniture_orders WHERE webhook_url IS NOT NULL AND webhook_url != '' ORDER BY created_at DESC LIMIT 1");
                    if ($stmt) {
                        $order = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if ($order && !empty($order['webhook_url'])) {
                            // Valideer dat de URL uit de database correct is (bevat AdalyaPoint pad op localhost)
                            $webhookFromDb = $order['webhook_url'];
                            $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
                            
                            // Als localhost en URL bevat NIET AdalyaPoint, skip deze oude URL
                            if ($host === 'localhost' && strpos($webhookFromDb, 'AdalyaPoint') === false && strpos($webhookFromDb, 'Dashboard') === false) {
                                // Oude/verkeerde URL in database, gebruik fallback
                            } else {
                                return $webhookFromDb;
                            }
                        }
                    }
                }
            }
        } catch (Exception $e) {
            // Database connectie gefaald, gebruik fallback (stilzwijgend)
        } catch (Error $e) {
            // Fatal error voorkomen, gebruik fallback (stilzwijgend)
        } catch (Throwable $e) {
            // Alle andere errors, gebruik fallback (stilzwijgend)
        }
    }
    
    // Als geen order gevonden, probeer te detecteren op basis van huidige request
    // Detecteer protocol (http of https)
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || 
              (!empty($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443) 
              ? 'https' : 'http';
    
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    // Check eerst environment variable (voor productie met aparte VPS servers)
    $vps1WebhookUrl = getenv('VPS1_WEBHOOK_URL');
    if ($vps1WebhookUrl) {
        return $vps1WebhookUrl;
    }
    
    // Als VPS1 en VPS2 op hetzelfde domein draaien (bijv. localhost),
    // gebruik hetzelfde domein voor de webhook URL maar met Dashboard 1 pad
    
    // Op localhost tijdens development met folders
    if ($host === 'localhost') {
        // Dashboard 1 (AdalyaPoint Admin) webhook URL
        return $scheme . '://' . $host . '/AdalyaPoint/Admin/public/api/webhook/payment-status';
    }
    
    // Op productie op hetzelfde domein (zonder dashboard folders)
    return $scheme . '://' . $host . '/api/webhook/payment-status';
    }
}

return [
    'name' => 'VPS2 Furniture Shop',
    'url' => detectBaseUrl(), // Automatisch gedetecteerd op basis van huidige request
    'timezone' => 'Europe/Amsterdam',
    'locale' => 'nl_NL',
    
    // Security
    'session_lifetime' => 120, // minuten
    'session_name' => 'vps2_furniture_session',
    
    // Development
    'debug' => true,
    'environment' => 'development', // development, production
    
    // VPS1 Webhook URL (waar payment updates naartoe gestuurd worden)
    // Wordt automatisch gedetecteerd op basis van VPS1's base URL uit API credentials
    'vps1_webhook_url' => detectVps1WebhookUrl(),
];

