<?php
namespace Core;

/**
 * Base Controller Class
 * Alle controllers erven van deze class
 */
abstract class Controller
{
    protected Database $db;
    protected Request $request;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->request = new Request();
    }

    /**
     * View renderen met data
     */
    protected function view(string $view, array $data = []): void
    {
        extract($data);
        
        $viewPath = __DIR__ . '/../app/Views/' . str_replace('.', '/', $view) . '.php';
        
        if (!file_exists($viewPath)) {
            die("View niet gevonden: {$view}");
        }
        
        require_once $viewPath;
    }

    /**
     * JSON response sturen
     */
    protected function json(array $data, int $statusCode = 200): void
    {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }

    /**
     * Redirect naar URL
     */
    protected function redirect(string $path): void
    {
        $baseUrl = rtrim($this->getBaseUrl(), '/');
        header("Location: {$baseUrl}{$path}");
        exit;
    }

    /**
     * Redirect terug met errors
     */
    protected function back(array $errors = []): void
    {
        if (!empty($errors)) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old'] = $this->request->all();
        }
        
        $referer = $_SERVER['HTTP_REFERER'] ?? $this->getBaseUrl();
        header("Location: {$referer}");
        exit;
    }

    /**
     * Flash message instellen
     */
    protected function flash(string $type, string $message): void
    {
        $_SESSION['flash'][$type] = $message;
    }

    /**
     * Check of gebruiker is ingelogd
     */
    protected function isAuthenticated(): bool
    {
        return isset($_SESSION['user_id']);
    }

    /**
     * Require authentication
     */
    protected function requireAuth(): void
    {
        if (!$this->isAuthenticated()) {
            $this->flash('error', 'Je moet ingelogd zijn om deze pagina te bekijken.');
            $this->redirect('/login');
        }
    }

    /**
     * Huidige gebruiker ophalen (MET CACHING)
     */
    protected function user(): ?array
    {
        if (!$this->isAuthenticated()) {
            return null;
        }
        
        // Check cache in session
        if (isset($_SESSION['user_data']) && $_SESSION['user_data']['id'] === $_SESSION['user_id']) {
            return $_SESSION['user_data'];
        }
        
        // Haal user data op uit database
        $user = $this->db->queryOne(
            "SELECT id, username, email, role, created_at FROM users WHERE id = ?",
            [$_SESSION['user_id']]
        );
        
        // Cache in session
        if ($user) {
            $_SESSION['user_data'] = $user;
        }
        
        return $user;
    }

    /**
     * Base URL ophalen
     */
    private function getBaseUrl(): string
    {
        $config = require __DIR__ . '/../config/app.php';
        return $config['url'];
    }
}

