<?php
namespace Core;

use PDO;
use PDOException;

/**
 * Database Class - Singleton Pattern
 * Beheert database connecties en queries
 */
class Database
{
    private static ?Database $instance = null;
    private ?PDO $connection = null;
    private array $config;

    /**
     * Private constructor voor Singleton
     */
    private function __construct()
    {
        $this->config = require __DIR__ . '/../config/database.php';
        $this->connect();
    }

    /**
     * Singleton instance ophalen
     */
    public static function getInstance(): Database
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Database connectie maken
     */
    private function connect(): void
    {
        try {
            // Probeer eerst via socket te verbinden (voor localhost)
            $socketPaths = ['/var/run/mysqld/mysqld.sock', '/tmp/mysql.sock', '/var/lib/mysql/mysql.sock'];
            $dsn = null;
            
            foreach ($socketPaths as $socketPath) {
                if (file_exists($socketPath)) {
                    $dsn = "mysql:unix_socket={$socketPath};dbname={$this->config['database']};charset={$this->config['charset']}";
                    break;
                }
            }
            
            // Als socket niet werkt, gebruik TCP
            if (!$dsn) {
                $dsn = "mysql:host={$this->config['host']};port={$this->config['port']};dbname={$this->config['database']};charset={$this->config['charset']}";
            }
            
            $this->connection = new PDO(
                $dsn,
                $this->config['username'],
                $this->config['password'],
                $this->config['options']
            );
        } catch (PDOException $e) {
            die("Database connectie mislukt: " . $e->getMessage());
        }
    }

    /**
     * PDO connectie ophalen
     */
    public function getConnection(): PDO
    {
        return $this->connection;
    }

    /**
     * Query uitvoeren (SELECT)
     */
    public function query(string $sql, array $params = []): array
    {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "doesn't exist") !== false || 
                strpos($e->getMessage(), "Unknown table") !== false) {
                error_log("Database Warning: " . $e->getMessage());
                return [];
            }
            error_log("Database Error: " . $e->getMessage());
            if ($this->config['debug'] ?? false) {
                throw $e;
            }
            return [];
        }
    }

    /**
     * Single row ophalen
     */
    public function queryOne(string $sql, array $params = []): ?array
    {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result ?: null;
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "doesn't exist") !== false || 
                strpos($e->getMessage(), "Unknown table") !== false) {
                error_log("Database Warning: " . $e->getMessage());
                return null;
            }
            error_log("Database Error: " . $e->getMessage());
            if ($this->config['debug'] ?? false) {
                throw $e;
            }
            return null;
        }
    }

    /**
     * Query uitvoeren (INSERT, UPDATE, DELETE)
     */
    public function execute(string $sql, array $params = []): bool
    {
        try {
            $stmt = $this->connection->prepare($sql);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "doesn't exist") !== false || 
                strpos($e->getMessage(), "Unknown table") !== false) {
                error_log("Database Warning: " . $e->getMessage());
                return false;
            }
            error_log("Database Error: " . $e->getMessage());
            if ($this->config['debug'] ?? false) {
                throw $e;
            }
            return false;
        }
    }

    /**
     * Laatste inserted ID ophalen
     */
    public function lastInsertId(): string
    {
        return $this->connection->lastInsertId();
    }

    /**
     * Transaction starten
     */
    public function beginTransaction(): bool
    {
        return $this->connection->beginTransaction();
    }

    /**
     * Transaction committen
     */
    public function commit(): bool
    {
        return $this->connection->commit();
    }

    /**
     * Transaction rollbacken
     */
    public function rollback(): bool
    {
        return $this->connection->rollBack();
    }

    /**
     * Voorkom cloning van singleton
     */
    private function __clone() {}

    /**
     * Voorkom unserialize van singleton
     */
    public function __wakeup()
    {
        throw new \Exception("Cannot unserialize singleton");
    }
}

