<?php
namespace Core;

/**
 * File Upload Helper
 * Beheert file uploads veilig
 */
class FileUpload
{
    private const MAX_FILE_SIZE = 5 * 1024 * 1024; // 5MB
    private const ALLOWED_TYPES = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    private const UPLOAD_DIR = __DIR__ . '/../public/uploads/products/';

    /**
     * Upload een image bestand
     */
    public static function uploadImage(array $file, string $prefix = 'product'): ?string
    {
        // Check of er een file is
        if (!isset($file['tmp_name']) || $file['error'] !== UPLOAD_ERR_OK) {
            return null;
        }

        // Check file size
        if ($file['size'] > self::MAX_FILE_SIZE) {
            throw new \Exception('Bestand is te groot. Max. grootte: 5MB');
        }

        // Check file type
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);

        if (!in_array($mimeType, self::ALLOWED_TYPES)) {
            throw new \Exception('Ongeldig bestandstype. Toegestaan: JPG, PNG, GIF, WEBP');
        }

        // Genereer unieke bestandsnaam
        $extension = self::getExtensionFromMime($mimeType);
        $filename = $prefix . '_' . uniqid() . '_' . time() . '.' . $extension;

        // Zorg dat upload directory bestaat
        if (!is_dir(self::UPLOAD_DIR)) {
            mkdir(self::UPLOAD_DIR, 0755, true);
        }

        // Verplaats bestand
        $destination = self::UPLOAD_DIR . $filename;
        
        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            throw new \Exception('Bestand kon niet worden opgeslagen');
        }

        // Return relatief pad voor database
        return 'uploads/products/' . $filename;
    }

    /**
     * Verwijder een image bestand
     */
    public static function deleteImage(string $imagePath): bool
    {
        if (empty($imagePath)) {
            return false;
        }

        // Als het al een relatief pad is, voeg public/ toe
        $fullPath = strpos($imagePath, 'public/') === 0 
            ? __DIR__ . '/../' . $imagePath 
            : __DIR__ . '/../public/' . $imagePath;

        if (file_exists($fullPath)) {
            return unlink($fullPath);
        }

        return false;
    }

    /**
     * Get extension from MIME type
     */
    private static function getExtensionFromMime(string $mimeType): string
    {
        $extensions = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp'
        ];

        return $extensions[$mimeType] ?? 'jpg';
    }
}
