<?php
namespace Core;

/**
 * Base Model Class
 * Alle models erven van deze class
 */
abstract class Model
{
    protected Database $db;
    protected string $table;
    protected string $primaryKey = 'id';

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * Alle records ophalen
     */
    public function all(): array
    {
        return $this->db->query("SELECT * FROM {$this->table}");
    }

    /**
     * Record ophalen op ID
     */
    public function find(int $id): ?array
    {
        return $this->db->queryOne(
            "SELECT * FROM {$this->table} WHERE {$this->primaryKey} = ?",
            [$id]
        );
    }

    /**
     * Record zoeken op voorwaarde
     */
    public function where(string $column, $value): ?array
    {
        return $this->db->queryOne(
            "SELECT * FROM {$this->table} WHERE {$column} = ?",
            [$value]
        );
    }

    /**
     * Meerdere records zoeken op voorwaarde
     */
    public function whereAll(string $column, $value): array
    {
        return $this->db->query(
            "SELECT * FROM {$this->table} WHERE {$column} = ?",
            [$value]
        );
    }

    /**
     * Record aanmaken
     */
    public function create(array $data): ?string
    {
        $columns = implode(', ', array_keys($data));
        $placeholders = implode(', ', array_fill(0, count($data), '?'));
        
        $sql = "INSERT INTO {$this->table} ({$columns}) VALUES ({$placeholders})";
        
        if ($this->db->execute($sql, array_values($data))) {
            return $this->db->lastInsertId();
        }
        
        return null;
    }

    /**
     * Record updaten
     */
    public function update(int $id, array $data): bool
    {
        $sets = [];
        $values = [];
        
        foreach ($data as $column => $value) {
            $sets[] = "{$column} = ?";
            $values[] = $value;
        }
        
        $values[] = $id;
        
        $sql = "UPDATE {$this->table} SET " . implode(', ', $sets) . " WHERE {$this->primaryKey} = ?";
        
        return $this->db->execute($sql, $values);
    }

    /**
     * Record verwijderen
     */
    public function delete(int $id): bool
    {
        return $this->db->execute(
            "DELETE FROM {$this->table} WHERE {$this->primaryKey} = ?",
            [$id]
        );
    }

    /**
     * Aantal records tellen
     */
    public function count(): int
    {
        $result = $this->db->queryOne("SELECT COUNT(*) as count FROM {$this->table}");
        return (int) ($result['count'] ?? 0);
    }

    /**
     * Custom query
     */
    protected function query(string $sql, array $params = []): array
    {
        return $this->db->query($sql, $params);
    }

    /**
     * Custom query (single result)
     */
    protected function queryOne(string $sql, array $params = []): ?array
    {
        return $this->db->queryOne($sql, $params);
    }
}

