/**
 * Vape Shop Dashboard - Main JavaScript
 * Moderne interactieve functionaliteit
 */

(function() {
    'use strict';

    // ========== DOCUMENT READY ==========
    document.addEventListener('DOMContentLoaded', function() {
        initializeApp();
    });

    // ========== INITIALIZE APP ==========
    function initializeApp() {
        initTableSearch();
        initFilterStatus();
        initAlertAutoDismiss();
        initFormValidation();
        initDeleteConfirmation();
    }

    // ========== TABLE SEARCH FUNCTIONALITY ==========
    function initTableSearch() {
        // Products search
        const searchProducts = document.getElementById('searchProducts');
        if (searchProducts) {
            searchProducts.addEventListener('keyup', function() {
                filterTable('productsTable', this.value);
            });
        }

        // Customers search
        const searchCustomers = document.getElementById('searchCustomers');
        if (searchCustomers) {
            searchCustomers.addEventListener('keyup', function() {
                filterTable('customersTable', this.value);
            });
        }
    }

    // ========== FILTER TABLE ==========
    function filterTable(tableId, searchValue) {
        const table = document.getElementById(tableId);
        if (!table) return;

        const tbody = table.querySelector('tbody');
        const rows = tbody.getElementsByTagName('tr');
        const search = searchValue.toLowerCase();

        for (let row of rows) {
            let text = row.textContent || row.innerText;
            text = text.toLowerCase();
            
            if (text.indexOf(search) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }

    // ========== STATUS FILTER FOR ORDERS ==========
    function initFilterStatus() {
        const filterStatus = document.getElementById('filterStatus');
        if (!filterStatus) return;

        filterStatus.addEventListener('change', function() {
            const selectedStatus = this.value.toLowerCase();
            const table = document.getElementById('ordersTable');
            if (!table) return;

            const tbody = table.querySelector('tbody');
            const rows = tbody.getElementsByTagName('tr');

            for (let row of rows) {
                const status = row.getAttribute('data-status');
                
                if (selectedStatus === '' || status === selectedStatus) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            }
        });
    }

    // ========== AUTO DISMISS ALERTS ==========
    function initAlertAutoDismiss() {
        const alerts = document.querySelectorAll('.alert');
        
        alerts.forEach(function(alert) {
            // Auto dismiss na 5 seconden
            setTimeout(function() {
                alert.style.animation = 'slideUp 0.3s ease';
                setTimeout(function() {
                    alert.remove();
                }, 300);
            }, 5000);

            // Klik om te sluiten
            alert.style.cursor = 'pointer';
            alert.addEventListener('click', function() {
                this.style.animation = 'slideUp 0.3s ease';
                setTimeout(function() {
                    alert.remove();
                }, 300);
            });
        });
    }

    // ========== FORM VALIDATION ==========
    function initFormValidation() {
        const forms = document.querySelectorAll('form');
        
        forms.forEach(function(form) {
            form.addEventListener('submit', function(e) {
                const requiredFields = form.querySelectorAll('[required]');
                let isValid = true;

                requiredFields.forEach(function(field) {
                    if (!field.value.trim()) {
                        isValid = false;
                        field.classList.add('error');
                        
                        // Remove error class on input
                        field.addEventListener('input', function() {
                            this.classList.remove('error');
                        });
                    }
                });

                if (!isValid) {
                    e.preventDefault();
                    showNotification('Vul alle verplichte velden in.', 'error');
                }
            });
        });
    }

    // ========== DELETE CONFIRMATION ==========
    function initDeleteConfirmation() {
        const deleteForms = document.querySelectorAll('form[action*="/delete/"]');
        
        deleteForms.forEach(function(form) {
            form.addEventListener('submit', function(e) {
                const confirmed = confirm('Weet je zeker dat je dit wilt verwijderen? Deze actie kan niet ongedaan worden gemaakt.');
                
                if (!confirmed) {
                    e.preventDefault();
                }
            });
        });
    }

    // ========== SHOW NOTIFICATION (Helper Function) ==========
    function showNotification(message, type = 'info') {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.innerHTML = `
            <i class="fas fa-${type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            ${message}
        `;

        const contentWrapper = document.querySelector('.content-wrapper');
        if (contentWrapper) {
            contentWrapper.insertBefore(alertDiv, contentWrapper.firstChild);
            
            // Auto dismiss
            setTimeout(function() {
                alertDiv.style.animation = 'slideUp 0.3s ease';
                setTimeout(function() {
                    alertDiv.remove();
                }, 300);
            }, 3000);
        }
    }

    // ========== ANIMATIONS ==========
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideUp {
            from {
                opacity: 1;
                transform: translateY(0);
            }
            to {
                opacity: 0;
                transform: translateY(-20px);
            }
        }
    `;
    document.head.appendChild(style);

    // ========== TABLE ROW HIGHLIGHT ==========
    const tables = document.querySelectorAll('.table tbody');
    tables.forEach(function(tbody) {
        const rows = tbody.querySelectorAll('tr');
        rows.forEach(function(row) {
            row.addEventListener('click', function(e) {
                // Skip if clicking on a button or link
                if (e.target.closest('button') || e.target.closest('a')) {
                    return;
                }
                
                // Remove highlight from all rows
                rows.forEach(function(r) {
                    r.style.backgroundColor = '';
                });
                
                // Highlight clicked row
                this.style.backgroundColor = '#f3f4f6';
                
                // Remove highlight after 2 seconds
                setTimeout(function() {
                    row.style.backgroundColor = '';
                }, 2000);
            });
        });
    });

    // ========== SMOOTH SCROLL ==========
    document.querySelectorAll('a[href^="#"]').forEach(function(anchor) {
        anchor.addEventListener('click', function(e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });

})();

