<?php
/**
 * Database Setup Script
 * Controleert en maakt de database aan indien nodig
 */

require_once __DIR__ . '/config/database.php';

$config = require __DIR__ . '/config/database.php';

echo "=== Database Setup Script ===\n\n";

try {
    // Probeer eerst via socket te verbinden (voor localhost)
    $pdo = null;
    $socketPaths = ['/var/run/mysqld/mysqld.sock', '/tmp/mysql.sock', '/var/lib/mysql/mysql.sock'];
    
    foreach ($socketPaths as $socketPath) {
        if (file_exists($socketPath)) {
            try {
                $dsn = "mysql:unix_socket={$socketPath};charset={$config['charset']}";
                $pdo = new PDO($dsn, $config['username'], $config['password'], [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
                ]);
                echo "✓ Verbonden via socket: $socketPath\n";
                break;
            } catch (PDOException $e) {
                // Probeer volgende socket
            }
        }
    }
    
    // Als socket niet werkt, probeer via TCP
    if (!$pdo) {
        $dsn = "mysql:host={$config['host']};port={$config['port']};charset={$config['charset']}";
        $pdo = new PDO($dsn, $config['username'], $config['password'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        echo "✓ Verbonden via TCP: {$config['host']}:{$config['port']}\n";
    }
    
    echo "✓ Verbonden met MySQL server\n";
    
    // Check of database bestaat
    $stmt = $pdo->query("SHOW DATABASES LIKE '{$config['database']}'");
    $exists = $stmt->rowCount() > 0;
    
    if ($exists) {
        echo "✓ Database '{$config['database']}' bestaat al\n";
        
        // Check of tabellen bestaan
        $pdo->exec("USE `{$config['database']}`");
        $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        
        if (count($tables) > 0) {
            echo "✓ Database bevat " . count($tables) . " tabel(len):\n";
            foreach ($tables as $table) {
                echo "  - $table\n";
            }
            echo "\n⚠ Database bestaat al met tabellen. Schema wordt NIET opnieuw uitgevoerd.\n";
            echo "   Als je de database opnieuw wilt aanmaken, verwijder eerst de database handmatig.\n";
        } else {
            echo "⚠ Database bestaat maar bevat geen tabellen. Schema wordt nu uitgevoerd...\n\n";
            $exists = false; // Forceer schema uitvoering
        }
    } else {
        echo "✗ Database '{$config['database']}' bestaat nog niet\n";
        echo "→ Database wordt nu aangemaakt...\n\n";
    }
    
    if (!$exists || (isset($tables) && count($tables) == 0)) {
        // Maak eerst de database aan
        if (!$exists) {
            echo "→ Database '{$config['database']}' aanmaken...\n";
            $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$config['database']}` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            echo "✓ Database aangemaakt\n";
        }
        
        // Gebruik nu de database
        $pdo->exec("USE `{$config['database']}`");
        echo "✓ Database geselecteerd\n\n";
        
        // Lees schema bestand
        $schemaFile = __DIR__ . '/database/vps2_schema.sql';
        if (!file_exists($schemaFile)) {
            throw new Exception("Schema bestand niet gevonden: $schemaFile");
        }
        
        $schema = file_get_contents($schemaFile);
        
        // Verwijder CREATE DATABASE en USE statements (die hebben we al gedaan)
        $schema = preg_replace('/CREATE DATABASE[^;]+;/i', '', $schema);
        $schema = preg_replace('/USE\s+[^;]+;/i', '', $schema);
        
        // Split in statements (eenvoudige split op ;)
        // Let op: dit werkt niet perfect voor complexe SQL, maar voor dit schema is het voldoende
        $statements = array_filter(
            array_map('trim', explode(';', $schema)),
            function($stmt) {
                return !empty($stmt) && 
                       !preg_match('/^--/', $stmt) && 
                       !preg_match('/^\/\*/', $stmt);
            }
        );
        
        // Voer schema uit met multi-statements support
        $pdo->setAttribute(PDO::MYSQL_ATTR_MULTI_STATEMENTS, true);
        $pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, 0);
        
        echo "→ Schema statements uitvoeren...\n";
        try {
            $pdo->exec($schema);
            echo "✓ Schema uitgevoerd\n";
        } catch (PDOException $e) {
            // Als er errors zijn, probeer statement voor statement
            echo "⚠ Directe import gaf waarschuwing, probeer statement voor statement...\n";
            foreach ($statements as $statement) {
                if (empty(trim($statement))) continue;
                
                try {
                    $pdo->exec($statement);
                } catch (PDOException $e2) {
                    // Skip errors voor IF NOT EXISTS statements als tabel al bestaat
                    if (strpos($e2->getMessage(), 'already exists') === false && 
                        strpos($e2->getMessage(), 'Duplicate') === false &&
                        strpos($e2->getMessage(), 'Duplicate entry') === false) {
                        // Alleen belangrijke errors tonen
                        if (strpos($e2->getMessage(), 'Unknown database') === false) {
                            echo "⚠ Waarschuwing: " . $e2->getMessage() . "\n";
                        }
                    }
                }
            }
            echo "✓ Schema uitgevoerd\n";
        }
        
        // Voer migrations uit
        echo "\n→ Migrations uitvoeren...\n";
        
        $migrations = [
            'add_image_url_column.sql',
            'add_categories_table.sql',
            'add_product_images_table.sql'
        ];
        
        // Zorg dat we nog steeds de juiste database gebruiken
        $pdo->exec("USE `{$config['database']}`");
        
        foreach ($migrations as $migration) {
            $migrationFile = __DIR__ . '/database/' . $migration;
            if (file_exists($migrationFile)) {
                $migrationSql = file_get_contents($migrationFile);
                $migrationStatements = array_filter(
                    array_map('trim', explode(';', $migrationSql)),
                    function($stmt) {
                        return !empty($stmt) && 
                               !preg_match('/^--/', $stmt) && 
                               !preg_match('/^\/\*/', $stmt);
                    }
                );
                
                foreach ($migrationStatements as $statement) {
                    if (empty(trim($statement))) continue;
                    try {
                        $pdo->exec($statement);
                    } catch (PDOException $e) {
                        // Skip errors voor IF NOT EXISTS
                        if (strpos($e->getMessage(), 'already exists') === false && 
                            strpos($e->getMessage(), 'Duplicate') === false &&
                            strpos($e->getMessage(), 'Duplicate entry') === false &&
                            strpos($e->getMessage(), 'Duplicate column') === false) {
                            echo "⚠ Waarschuwing bij migration $migration: " . $e->getMessage() . "\n";
                        }
                    }
                }
                echo "✓ Migration $migration uitgevoerd\n";
            }
        }
        
        // Check tabellen
        $pdo->exec("USE `{$config['database']}`");
        $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        echo "\n✓ Database setup voltooid!\n";
        echo "✓ Aantal tabellen: " . count($tables) . "\n";
        if (count($tables) > 0) {
            echo "✓ Tabellen:\n";
            foreach ($tables as $table) {
                echo "  - $table\n";
            }
        }
    }
    
    echo "\n=== Klaar! ===\n";
    
} catch (PDOException $e) {
    echo "✗ Database fout: " . $e->getMessage() . "\n";
    exit(1);
} catch (Exception $e) {
    echo "✗ Fout: " . $e->getMessage() . "\n";
    exit(1);
}
